<?php
/*
Plugin Name: Inten Star Rating Schema
Description: This plugin add star rating schema to any page of your site.
Version: 1.1
Author: Inten (Rasool Vahdati)
Author URI: https://inten.asia
Text Domain: inten_star_rating
*/

if (!defined('ABSPATH')) {
	exit;
} // Exit if accessed directly

if (!class_exists('Inten_Star_Rating'))
{
	class Inten_Star_Rating
	{

		/**
		 * Construct the plugin object
		 */
		public function __construct()
		{

			// define constants
			define( 'INTEN_STAR_RATING_PLUGIN_Version', '1.0' );
			define( 'INTEN_STAR_RATING_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
			define( 'INTEN_STAR_RATING_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

			// add text domain 
			add_action( 'plugins_loaded', array( &$this , 'load_plugin_textdomain' ) );

			// create api rest route
			add_action( 'admin_notices' , array( &$this , 'star_rating_admin_notice' ) );

			// add activate code for getting license
			add_action( 'admin_footer' , array( &$this , 'star_rating_admin_footer_scripts' ) );

			// ajax for activating plugin
			add_action( 'wp_ajax_activate_inten_plugin', array( &$this , 'ajax_activate_inten_pkugin' ) );

			// add header script
			add_action( 'wp_head' , array( &$this , 'star_rating_header_scripts' ) );

			// add settings
			add_action( 'admin_init', array(&$this, 'inten_star_settings_init'));
            add_action( 'admin_menu', array(&$this, 'inten_star_settings_submenu'));

		} // END public function __construct()

		public function load_plugin_textdomain()
		{
			load_plugin_textdomain( 'inten_star_rating', FALSE, basename( dirname( __FILE__ ) ) . '/languages/' );
		}

		public function star_rating_admin_notice()
		{
			$inten_star_rating_token = get_option('inten_star_rating_token');
			if(empty($inten_star_rating_token)) : ?>
			<div class="notice notice-success is-dismissible">
				<p>
					<?php _e( 'Inten Star Rating Schema is not active. Please fill below form to activate the plugin.', 'inten_star_rating' ); ?>
					<p class="star-result"></p>
					<form action="" class="star-form" method="post">
						<label for="star-email"><?php _e('Email: ', 'inten_star_rating'); ?></label>
						<input type="email" name="star-email" id="star-email" />
						<label for="star-mobile"><?php _e('Mobile: ','inten_star_rating'); ?></label>
						<input type="tel" name="star-mobile" id="star-mobile" />
						<input type="submit" class="button button-primary activate-plugin" value="<?php _e('Activate', 'inten_star_rating'); ?>">
					</form>
				</p>
			</div>
			<?php
			endif;
		}

		public function star_rating_admin_footer_scripts()
		{
			$inten_star_rating_token = get_option('inten_star_rating_token');
			if(empty($inten_star_rating_token)) : ?>
			<script>
			jQuery(document).ready(function(){
				jQuery('.activate-plugin').on('click', function(e){
					e.preventDefault();
					var activate_selector = jQuery(this);
					var star_email = jQuery('#star-email').val();
					var star_mobile = jQuery('#star-mobile').val();
					jQuery.ajax({
						method: 'POST',
						url: '<?= admin_url('admin-ajax.php') ?>',
						data: { action: 'activate_inten_plugin' , email : star_email , mobile : star_mobile },
						dataType: 'json',
						beforeSend: function() {
							jQuery('.star-result').css('color', 'black').html('<?= _e('Please wait...','inten_star_rating') ?>');
						},
						success: function(response) {
							if(response.status=='ok')
							{
								activate_selector.parent().parent().html(response.msg);
							}
							else
							{
								jQuery('.star-result').css('color', 'red').html(response.msg);
							}
						},
						error: function(error)
						{
							console.log(error);
						}
					});
				})
			});
			</script>
			<?php
			endif;
		}

		/**
		 * Ajax handler for activate inten plugin
		 */
		public function ajax_activate_inten_pkugin()
		{
			$response = [];
			$email = sanitize_email( $_POST['email'] );
			$mobile = sanitize_text_field( $_POST['mobile'] );
			$web_url = site_url();
			
			// send to api
			$api_url  = 'https://inten.asia/wp-json/api/v1/reg-url/';
		
			// api arguments
			$args = array(
				'method'      => 'POST',
				'body'        => [
					'email' => $email,
					'mobile' => $mobile,
					'url' => $web_url
				]
			);
		
			$request = wp_remote_post( $api_url, $args );
			$result = json_decode( wp_remote_retrieve_body( $request ), true );

			if ( is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) != 200 )
			{
				$response = array('status' => 'no', 'msg' => $result['message'] );
			}
			else
			{
				if(!empty($result['token']))
				{
					update_option( 'inten_star_rating_token' , $result['token'] );
				}
				$response = array('status' => 'ok', 'msg' => __('<p>The plugin activated successfully.</p>', 'inten_star_rating' ) );
			}

			echo wp_json_encode( $response );
			wp_die();
		}

		/**
		 * Add code to header
		 */
		public function star_rating_header_scripts()
		{
			$inten_star_rating_token = get_option('inten_star_rating_token');
			$show_code = 1;
			if(!empty($inten_star_rating_token))
			{
				$current_id = get_the_ID();
				if(empty($current_id))
				{
					$current_id = 0;
				}
				else
				{
					$single_post_type = get_post_type( get_the_ID() );
					$inten_star_rating_options = get_option( 'inten_star_rating_options' );
					if(!in_array($single_post_type, $inten_star_rating_options['post_type']))
						$show_code = 0;
				}
				?>

				<?php if( $show_code ) : ?>
				<script type="application/ld+json">
				[{
					"@context": "https://schema.org/",
					"@type": "CreativeWorkSeason",
					"name": "<?php wp_title(); ?>",
					"aggregateRating": {
					"@type": "AggregateRating",
					"ratingValue": "4.<?php echo substr(get_the_ID() , 0 , 1)?>",
					"ratingCount": "<?php echo get_the_ID()?>",
					"bestRating": "5",
					"worstRating": "1"
				}}]
				</script>
				<?php endif; ?>

			<?php
			}
		}

		public function inten_star_settings_init()
        {
            // Register a new setting for "wporg" page.
            register_setting( 'inten_star_rating', 'inten_star_rating_options' );
        
            // Register a new section in the "wporg" page.
            add_settings_section( 'inten_star_rating_section', '', array(&$this, 'inten_star_rating_section_callback') , 'inten_star_rating' );
        
            // Register a new field in the "wporg_section_developers" section, inside the "wporg" page.
            add_settings_field(
                'inten_star_rating_post_type', // As of WP 4.6 this value is used only internally.
                __('Post Types', 'inten_star_rating'),
                array(&$this, 'inten_star_rating_post_type_callback'),
                'inten_star_rating',
                'inten_star_rating_section',
                array(
                    'label_for'    => 'post_type',
                    'class'        => 'inten_star_rating_post_type',
                )
            );
        }

        /**
         * Developers section callback function.
         *
         * @param array $args  The settings array, defining title, id, callback.
         */
        public function inten_star_rating_section_callback( $args ) {
            
        }

        /**
         *
         * Post type callback
         *
         * @param array $args
         */
        public function inten_star_rating_post_type_callback( $args ) {
            // Get the value of the setting we've registered with register_setting()
            $options = get_option( 'inten_star_rating_options' );
			$post_types = get_post_types(['public'   => true ], 'objects');
			if(!empty($post_types)) :
			?>
				<?php foreach($post_types as $post_type) : ?>
					<?php if ( isset( $post_type->labels->name ) ) : ?>
					<input type="checkbox" id="<?php echo esc_attr( $args['class'] ); ?>" class="<?php echo esc_attr( $args['class'] ); ?>" name="inten_star_rating_options[<?php echo esc_attr( $args['label_for'] ); ?>][]" value="<?php echo esc_attr( $post_type->name ); ?>" <?php if( isset( $options[ $args['label_for'] ] ) && in_array( $post_type->name, $options[ $args['label_for'] ] ) ) echo 'checked="checked"'; ?>> <?php echo $post_type->labels->name; ?><br />
					<?php endif; ?>
				<?php endforeach; ?>
            <p class="description"><?php echo __('Please check your custom post types in top list to show schema in single page of them.', 'inten_star_rating'); ?></p>
            <?php
			endif;
        }

        /**
         * Add the top level menu page.
         */
        public function inten_star_settings_submenu()
        {
			$inten_star_rating_token = get_option('inten_star_rating_token');
			if(!empty($inten_star_rating_token))
			{
				add_submenu_page(
					'options-general.php',
					__('Inten Star Rating Schema Settings', 'inten_star_rating'),
					__('Inten Star Rating Schema', 'inten_star_rating'),
					'manage_options',
					'inten-star-settings',
					array(&$this, 'inten_star_settings_options_html')
				);
			}
        }

        /**
         * Top level menu callback function
         */
        public function inten_star_settings_options_html()
		{
			
			$inten_star_rating_token = get_option('inten_star_rating_token');
			if(!empty($inten_star_rating_token))
			{
				// check user capabilities
				if ( ! current_user_can( 'manage_options' ) ) {
					return;
				}

				// add error/update messages

				// check if the user have submitted the settings
				// WordPress will add the "settings-updated" $_GET parameter to the url
				if ( isset( $_GET['settings-updated'] ) ) {
					// add settings saved message with the class of "updated"
					//add_settings_error( 'inten_star_rating_messages', 'inten_star_rating_message', __('Settings saved.', 'inten_star_rating') , 'updated' );
				}

				// show error/update messages
				settings_errors( 'inten_star_rating_messages' );
				?>
				<div class="wrap">
					<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
					<form action="options.php" method="post">
						<?php
						// output security fields for the registered setting "wporg"
						settings_fields( 'inten_star_rating' );
						// output setting sections and their fields
						// (sections are registered for "wporg", each field is registered to a specific section)
						do_settings_sections( 'inten_star_rating' );
						// output save settings button
						submit_button( __('Save Settings', 'inten_star_rating') );
						?>
					</form>
				</div>
				<?php
			}
            
        }

		/**
		 * Activate the plugin
		 */
		public static function activate()
		{
			// Do nothing
		} // END public static function activate

		/**
		 * Deactivate the plugin
		 */
		public static function deactivate()
		{
			// Do nothing
		} // END public static function deactivate	

	} // END class Inten_Star_Rating
} // END if(!class_exists('Inten_Star_Rating'))

if (class_exists('Inten_Star_Rating')) {
	// instantiate the plugin class
	new Inten_Star_Rating();

	register_activation_hook( __FILE__, array( 'Inten_Star_Rating', 'activate' ) );
	register_deactivation_hook( __FILE__, array( 'Inten_Star_Rating', 'deactivate' ) );
}